@extends('layouts.app')

@section('title', 'Add Product Model - Quotation App')

@section('styles')
<style>
    .form-container { max-width: 900px; }
    
    /* Navigation breadcrumb */
    .nav-breadcrumb { 
        background: #f3f4f6; 
        padding: 12px; 
        border-radius: 6px; 
        margin-bottom: 16px; 
        font-size: 13px;
        display: flex;
        gap: 8px;
        align-items: center;
        flex-wrap: wrap;
    }
    .nav-breadcrumb a { color: #2563eb; text-decoration: none; font-weight: 500; }
    .nav-breadcrumb a:hover { text-decoration: underline; }
    .nav-breadcrumb span { color: #6b7280; }
    
    /* Master selector card */
    .master-selector { 
        background: #ffffffff;
        padding: 24px; 
        border-radius: 8px; 
        margin-bottom: 24px;
        color: white;
        border: 2px solid #9ca3af;
    }
    .master-selector h2 { margin: 0 0 16px 0; font-size: 18px; }
    .master-selector .form-group { margin-bottom: 12px; }
    .master-selector label { color: rgba(255,255,255,0.9); }
    .master-selector select { 
        background: white;
        color: #374151;
        border: none;
        padding: 12px;
        border-radius: 6px;
        font-weight: 500;
    }
    
    /* Master info display */
    .master-info-card {
        background: #f0fdf4;
        border: 2px solid #22c55e;
        border-radius: 8px;
        padding: 16px;
        margin-bottom: 24px;
        display: none;
    }
    .master-info-card.active { display: block; }
    .master-info-card h3 { margin: 0 0 12px 0; color: #15803d; font-size: 16px; }
    .master-info-content { display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 16px; }
    .info-item { font-size: 13px; }
    .info-item strong { display: block; color: #374151; margin-bottom: 4px; }
    .info-item span { color: #6b7280; }
    
    .form-group { margin-bottom: 16px; }
    .form-group label { display: block; margin-bottom: 6px; font-weight: 500; color: #374151; font-size: 14px; }
    .form-group label span { color: #dc2626; }
    .form-control, .form-select { width: 100%; padding: 10px 12px; border: 1px solid #d1d5db; border-radius: 6px; font-size: 14px; }
    .form-control:focus, .form-select:focus { outline: none; border-color: #2b2b2bff; box-shadow: 0 0 0 3px rgba(102,126,234,0.1); }
    
    .btn-group { display: flex; gap: 12px; margin-top: 24px; }
    .section-title { font-size: 16px; font-weight: 600; color: #374151; margin: 24px 0 16px; padding-top: 16px; border-top: 1px solid #e5e7eb; }
    
    .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; }
    .form-row-3 { display: grid; grid-template-columns: 2fr 1fr 1fr; gap: 12px; }
    
    .spec-row { display: grid; grid-template-columns: 2fr 1fr 1fr auto; gap: 12px; margin-bottom: 12px; align-items: end; }
    .spec-row .form-group { margin-bottom: 0; }
    .spec-header { display: grid; grid-template-columns: 2fr 1fr 1fr auto; gap: 12px; margin-bottom: 8px; }
    .spec-header span { font-size: 13px; font-weight: 500; color: #6b7280; }
    
    #specs-container { margin-bottom: 16px; }
    .btn-remove { padding: 8px 12px; background: #ef4444; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 12px; }
    .btn-remove:hover { background: #dc2626; }
    .btn-add { padding: 8px 16px; background: #059669; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 12px; }
    .btn-add:hover { background: #047857; }
    
    .disabled-section {
        opacity: 0.5;
        pointer-events: none;
    }
    
    @media (max-width: 600px) { 
        .form-row { grid-template-columns: 1fr; }
        .form-row-3 { grid-template-columns: 1fr; }
        .spec-row { grid-template-columns: 1fr; }
        .master-info-content { grid-template-columns: 1fr; }
        .spec-header { display: none; }
    }
</style>
@endsection

@section('content')
<div class="nav-breadcrumb">
    <span>🔧 Product Models</span>
    <span>/</span>
    <a href="{{ route('products.index') }}">Back to Models</a>
    <span>•</span>
    <span style="color: #374151; font-weight: 500;">Add New</span>
    <span>|</span>
    <a href="{{ route('master.index') }}">View Main Products</a>
</div>

<div class="form-container">
    <div class="card">
        <div class="card-header">
            <h1>Add Product Model</h1>
            <p style="color: #6b7280; font-size: 14px; margin: 8px 0 0 0;">Select a main product, then add a new model with specifications and pricing</p>
        </div>
        <div class="card-body">
            @if($errors->any())
                <div class="alert alert-danger">
                    <ul style="margin-left: 20px;">
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <form action="{{ route('products.store') }}" method="POST">
                @csrf
                
                <!-- STEP 1: SELECT MASTER PRODUCT -->
                <div class="master-selector">
                    <h2>📦 Step 1: Select Main Product</h2>
                    <div class="form-group">
                        <label for="product_master_id" style="color: rgba(0, 0, 0, 0.9);">Main Product <span>*</span></label>
                        <select class="form-select" style="border: 2px solid #9ca3af;" id="product_master_id" name="product_master_id" required onchange="loadMasterDetails()">
                            <option value="">-- Choose a Main Product --</option>
                            @foreach($productMasters as $master)
                                <option value="{{ $master->id }}" {{ old('product_master_id') == $master->id ? 'selected' : '' }}>
                                    {{ $master->product_name }}
                                </option>
                            @endforeach
                        </select>
                        @error('product_master_id')<small style="color: rgba(255,255,255,0.9);">{{ $message }}</small>@enderror
                    </div>
                </div>

                <!-- Master Info Display -->
                <div id="masterInfo" class="master-info-card">
                    <h3>✓ Main Product Info</h3>
                    <div class="master-info-content">
                        <div class="info-item">
                            <strong>Note:</strong>
                            <span id="masterNote">-</span>
                        </div>
                        <div class="info-item">
                            <strong>Standard Accessories:</strong>
                            <span id="masterStd">-</span>
                        </div>
                        <div class="info-item">
                            <strong>Optional Accessories:</strong>
                            <span id="masterOpt">-</span>
                        </div>
                    </div>
                </div>

                <!-- STEP 2: ADD MODEL DETAILS -->
                <h3 class="section-title" id="step2-title" style="display: none;">🔧 Step 2: Model Details</h3>
                
                <div id="step2-content" class="disabled-section">
                    <!-- Toggle: Single vs Multiple Products -->
                    <div style="margin-bottom: 16px; display: flex; gap: 12px;">
                        <label style="display: flex; align-items: center; gap: 6px; cursor: pointer;">
                            <input type="radio" name="product_entry_mode" value="single" checked onchange="toggleProductMode('single')">
                            Single Product
                        </label>
                        <label style="display: flex; align-items: center; gap: 6px; cursor: pointer;">
                            <input type="radio" name="product_entry_mode" value="multiple" onchange="toggleProductMode('multiple')">
                            Multiple Products
                        </label>
                    </div>

                    <!-- SINGLE PRODUCT MODE -->
                    <div id="single-mode" style="display: block;">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="product_model">Model/Variant Name <span>*</span></label>
                                <input type="text" class="form-control" id="product_model" name="product_model" 
                                       value="{{ old('product_model') }}" placeholder="e.g. Model A, Variant X, Premium Edition">
                                @error('product_model')<small style="color: #dc2626;">{{ $message }}</small>@enderror
                            </div>
                            <div class="form-group">
                                <label for="price">Price <span>*</span></label>
                                <input type="number" step="0.01" class="form-control" id="price" name="price" 
                                       value="{{ old('price') }}" placeholder="0.00">
                                @error('price')<small style="color: #dc2626;">{{ $message }}</small>@enderror
                            </div>
                        </div>
                    </div>

                    <!-- MULTIPLE PRODUCTS MODE -->
                    <div id="multiple-mode" style="display: none;">
                        <h4 style="color: #374151; margin-bottom: 12px; font-size: 16px; font-weight: 600;">📦 Multiple Product Models with Specifications</h4>
                        
                        <button type="button" class="btn-add" id="loadTemplateBtn2" style="margin-bottom: 16px; font-size: 14px; padding: 8px 16px;">📋 Load Template from Master</button>

                        <!-- Specifications Table (Spreadsheet Style) -->
                        <div style="overflow-x: auto; margin-bottom: 12px; border: 2px solid #e5e7eb; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                            <table style="width: 100%; border-collapse: collapse; background: white; table-layout: fixed;">
                                <colgroup>
                                    <col style="width: 130px;">
                                    <!-- Spec columns will be added here dynamically (120px each) -->
                                    <col style="width: 110px;">
                                    <col style="width: 75px;">
                                </colgroup>
                                <thead>
                                    <tr id="header-row" style="background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); color: white; font-weight: 600;">
                                        <th style="padding: 14px 12px; text-align: left; border: 1px solid #cbd5e1; font-size: 13px; overflow: hidden; text-overflow: ellipsis;">Model Name</th>
                                        <!-- Spec columns will be added here dynamically -->
                                        <th style="padding: 14px 12px; text-align: left; border: 1px solid #cbd5e1; font-size: 13px; overflow: hidden; text-overflow: ellipsis;">Price (₹)</th>
                                        <th style="padding: 14px 12px; text-align: center; border: 1px solid #cbd5e1; font-size: 13px; overflow: hidden;">Action</th>
                                    </tr>
                                </thead>
                                <tbody id="products-table-body">
                                    <!-- Product rows will be added here -->
                                </tbody>
                            </table>
                        </div>

                        <button type="button" class="btn-add" onclick="addProductRowForTable()" style="font-size: 14px; padding: 8px 16px;">+ Add Another Model</button>
                    </div>

                    <!-- STEP 3: SPECIFICATIONS -->
                    <h3 class="section-title">📋 Step 3: Manage Specifications</h3>
                    <p style="font-size: 12px; color: #6b7280; margin-bottom: 12px;">In multiple mode, click "Load Template" above to set specification columns. Edit values directly in the table.</p>
                    
                    <button type="button" class="btn-add" id="loadTemplateBtn" style="margin-bottom: 12px; display: none;">📋 Load Template from Master</button>
                    
                    <div class="spec-header">
                        <span>Specification Name</span>
                        <span>Value</span>
                        <span>Unit</span>
                        <span></span>
                    </div>

                    <div id="specs-container">
                        <div class="spec-row">
                            <div class="form-group">
                                <input type="text" class="form-control" name="spec_name[]" placeholder="e.g. Cutting Length">
                            </div>
                            <div class="form-group">
                                <input type="text" class="form-control" name="spec_value[]" placeholder="e.g. 2540">
                            </div>
                            <div class="form-group">
                                <input type="text" class="form-control" name="spec_unit[]" placeholder="e.g. MM">
                            </div>
                            <div style="display: flex; gap: 4px; align-items: center;">
                                <button type="button" class="btn-remove" onclick="removeSpecRow(this)">Remove</button>
                            </div>
                        </div>
                    </div>

                    <button type="button" class="btn-add" onclick="addSpecRow()">+ Add Another Specification</button>
                </div>

                <div class="btn-group">
                    <button type="submit" class="btn btn-primary" id="submitBtn" disabled>Create Product Model(s)</button>
                    <a href="{{ route('products.index') }}" class="btn btn-secondary">Cancel</a>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    const masterData = {
        @foreach($productMasters as $master)
            "{{ $master->id }}": {
                "note": "{{ $master->note ?? '-' }}",
                "standard_accessories": "{{ implode(', ', $master->getStandardAccessoriesArray()) ?: 'None' }}",
                "optional_accessories": "{{ implode(', ', $master->getOptionalAccessoriesArray()) ?: 'None' }}"
            },
        @endforeach
    };

    function loadMasterDetails() {
        const masterId = document.getElementById('product_master_id').value;
        const masterInfo = document.getElementById('masterInfo');
        const step2Title = document.getElementById('step2-title');
        const step2Content = document.getElementById('step2-content');
        const submitBtn = document.getElementById('submitBtn');
        const loadTemplateBtn = document.getElementById('loadTemplateBtn');
        
        if (masterId && masterData[masterId]) {
            const data = masterData[masterId];
            document.getElementById('masterNote').textContent = data.note;
            document.getElementById('masterStd').textContent = data.standard_accessories;
            document.getElementById('masterOpt').textContent = data.optional_accessories;
            
            masterInfo.classList.add('active');
            step2Title.style.display = 'block';
            step2Content.classList.remove('disabled-section');
            submitBtn.disabled = false;
            loadTemplateBtn.style.display = 'block';
        } else {
            masterInfo.classList.remove('active');
            step2Title.style.display = 'none';
            step2Content.classList.add('disabled-section');
            submitBtn.disabled = true;
            loadTemplateBtn.style.display = 'none';
        }
    }

    // Load template from master product
    async function loadTemplate() {
        const masterId = document.getElementById('product_master_id').value;
        if (!masterId) {
            alert('Please select a master product first');
            return;
        }

        try {
            const response = await fetch(`/products/master/${masterId}/template`);
            const data = await response.json();

            if (data.template && data.template.length > 0) {
                // Clear existing specs
                clearSpecsContainer();

                // Add template specs with name and unit
                data.template.forEach((spec) => {
                    // spec is object: {name: "...", unit: "..."}
                    const specName = typeof spec === 'string' ? spec : (spec.name || '');
                    const specUnit = typeof spec === 'object' ? (spec.unit || '') : '';
                    addSpecRow(specName, specUnit);
                });

                alert(`Loaded ${data.count} specification${data.count !== 1 ? 's' : ''} from template!`);
            } else {
                alert('No template found for this product master. Please add specifications manually.');
            }
        } catch (error) {
            console.error('Error loading template:', error);
            alert('Failed to load template. Please try again.');
        }
    }

    function clearSpecsContainer() {
        const container = document.getElementById('specs-container');
        container.innerHTML = '';
    }

    function addSpecRow(specName = '', specUnit = '') {
        const container = document.getElementById('specs-container');
        const row = document.createElement('div');
        row.className = 'spec-row';
        row.innerHTML = `
            <div class="form-group">
                <input type="text" class="form-control" name="spec_name[]" placeholder="Specification Name" value="${specName}">
            </div>
            <div class="form-group">
                <input type="text" class="form-control" name="spec_value[]" placeholder="Value">
            </div>
            <div class="form-group">
                <input type="text" class="form-control" name="spec_unit[]" placeholder="Unit" value="${specUnit}">
            </div>
            <div style="display: flex; gap: 4px; align-items: center;">
                <button type="button" class="btn-remove" onclick="removeSpecRow(this)">Remove</button>
            </div>
        `;
        container.appendChild(row);
    }

    function removeSpecRow(btn) {
        btn.closest('.spec-row').remove();
    }

    // Toggle between single and multiple product modes
    function toggleProductMode(mode) {
        const singleMode = document.getElementById('single-mode');
        const multipleMode = document.getElementById('multiple-mode');
        
        if (mode === 'single') {
            singleMode.style.display = 'block';
            multipleMode.style.display = 'none';
        } else {
            singleMode.style.display = 'none';
            multipleMode.style.display = 'block';
        }
    }

    // Add product row in multiple mode
    function addProductRow() {
        const tbody = document.getElementById('products-table-body');
        const rowCount = tbody.children.length;
        
        const newRow = document.createElement('tr');
        newRow.className = 'product-row';
        newRow.innerHTML = `
            <td style="padding: 10px; border: 1px solid #ddd;">
                <input type="text" name="products[${rowCount}][product_model]" placeholder="e.g. AFBR-215" class="form-control" style="font-size: 13px;">
            </td>
            <td style="padding: 10px; border: 1px solid #ddd;">
                <input type="number" step="0.01" name="products[${rowCount}][price]" placeholder="0.00" class="form-control" style="font-size: 13px;">
            </td>
            <td style="padding: 10px; border: 1px solid #ddd; text-align: center;">
                <button type="button" class="btn-remove" onclick="removeProductRow(this)">Remove</button>
            </td>
        `;
        tbody.appendChild(newRow);
    }

    // Remove product row
    function removeProductRow(btn) {
        btn.closest('tr').remove();
    }

    // Store template specs globally
    let currentTemplateSpecs = [];

    // Add product row for table (with spec input fields)
    function addProductRowForTable() {
        const tbody = document.getElementById('products-table-body');
        const rowIndex = tbody.children.length;
        
        const row = document.createElement('tr');
        row.className = 'product-row';
        row.style.backgroundColor = rowIndex % 2 === 0 ? '#f8fafc' : 'white';
        row.style.borderBottom = '1px solid #e2e8f0';
        
        // MODEL NAME COLUMN
        let rowHTML = `
            <td style="padding: 12px; border: 1px solid #cbd5e1; overflow: hidden;">
                <input type="text" name="products[${rowIndex}][product_model]" placeholder="e.g. AFBR-212" 
                       class="form-control" style="font-size: 13px; padding: 8px; border: 1px solid #cbd5e1; border-radius: 4px; width: 100%; box-sizing: border-box;">
            </td>
        `;
        
        // SPEC COLUMNS - Must match header exactly (120px each)
        currentTemplateSpecs.forEach((spec, specIndex) => {
            rowHTML += `
                <td style="padding: 12px; border: 1px solid #cbd5e1; overflow: hidden;">
                    <input type="text" name="products[${rowIndex}][specs][${specIndex}][value]" 
                           placeholder="Value" class="form-control" 
                           style="font-size: 13px; padding: 8px; border: 1px solid #cbd5e1; border-radius: 4px; width: 100%; box-sizing: border-box;">
                    <input type="hidden" name="products[${rowIndex}][specs][${specIndex}][name]" value="${spec.name}">
                    <input type="hidden" name="products[${rowIndex}][specs][${specIndex}][unit]" value="${spec.unit}">
                </td>
            `;
        });
        
        // PRICE COLUMN
        rowHTML += `
            <td style="padding: 12px; border: 1px solid #cbd5e1; overflow: hidden;">
                <input type="number" step="0.01" name="products[${rowIndex}][price]" 
                       placeholder="0.00" class="form-control" 
                       style="font-size: 13px; padding: 8px; border: 1px solid #cbd5e1; border-radius: 4px; width: 100%; box-sizing: border-box;">
            </td>
        `;
        
        // ACTION COLUMN
        rowHTML += `
            <td style="padding: 12px; border: 1px solid #cbd5e1; text-align: center; overflow: hidden;">
                <button type="button" class="btn-remove" onclick="removeProductRow(this)" 
                        style="font-size: 12px; padding: 6px 12px; white-space: nowrap;">Remove</button>
            </td>
        `;
        
        row.innerHTML = rowHTML;
        tbody.appendChild(row);
    }

    // Load template and setup table columns
    async function loadTemplateForMultipleMode() {
        const masterId = document.getElementById('product_master_id').value;
        if (!masterId) {
            alert('Please select a master product first');
            return;
        }

        try {
            const response = await fetch(`/products/master/${masterId}/template`);
            const data = await response.json();

            console.log('Template Response:', data);  // DEBUG

            if (data.template && data.template.length > 0) {
                // Store template specs
                currentTemplateSpecs = data.template.map(spec => ({
                    name: typeof spec === 'string' ? spec : (spec.name || ''),
                    unit: typeof spec === 'object' ? (spec.unit || '') : ''
                }));

                console.log('Current Template Specs:', currentTemplateSpecs);  // DEBUG

                // Clear existing table and rebuild header
                rebuildTableHeader();
                
                // Clear existing rows
                const tbody = document.getElementById('products-table-body');
                tbody.innerHTML = '';
                
                // Add initial empty row
                addProductRowForTable();

                alert(`Template loaded! ${data.count} specification${data.count !== 1 ? 's' : ''} ready.`);
            } else {
                alert('No template found for this product master. Please define specifications in the Product Master.');
            }
        } catch (error) {
            console.error('Error loading template:', error);
            alert('Failed to load template. Please try again.');
        }
    }

    // Rebuild table header with spec columns
    function rebuildTableHeader() {
        const table = document.querySelector('#multiple-mode table');
        const thead = table.querySelector('thead');
        const tbody = table.querySelector('tbody');
        
        console.log('Rebuilding header with', currentTemplateSpecs.length, 'specs');
        
        // Remove old colgroup and create new one
        const oldColgroup = table.querySelector('colgroup');
        if (oldColgroup) oldColgroup.remove();
        
        const newColgroup = document.createElement('colgroup');
        
        // Add Model Name col
        let col = document.createElement('col');
        col.style.width = '130px';
        newColgroup.appendChild(col);
        
        // Add spec cols
        currentTemplateSpecs.forEach(spec => {
            col = document.createElement('col');
            col.className = 'spec-col';
            col.style.width = '120px';
            newColgroup.appendChild(col);
        });
        
        // Add Price col
        col = document.createElement('col');
        col.style.width = '110px';
        newColgroup.appendChild(col);
        
        // Add Action col
        col = document.createElement('col');
        col.style.width = '75px';
        newColgroup.appendChild(col);
        
        // Insert colgroup at start of table
        table.insertBefore(newColgroup, thead);
        
        // Rebuild header row
        let headerHTML = `
            <th style="padding: 14px 12px; text-align: left; border: 1px solid #cbd5e1; font-size: 13px;">Model Name</th>
        `;
        
        // Add spec headers
        currentTemplateSpecs.forEach(spec => {
            const unitText = spec.unit ? ` (${spec.unit})` : '';
            headerHTML += `<th style="padding: 14px 12px; text-align: left; border: 1px solid #cbd5e1; font-size: 13px;">${spec.name}${unitText}</th>`;
        });
        
        // Add Price and Action headers
        headerHTML += `
            <th style="padding: 14px 12px; text-align: left; border: 1px solid #cbd5e1; font-size: 13px;">Price (₹)</th>
            <th style="padding: 14px 12px; text-align: center; border: 1px solid #cbd5e1; font-size: 13px;">Action</th>
        `;
        
        // Update header row
        const headerRow = thead.querySelector('tr');
        headerRow.innerHTML = headerHTML;
        
        // Recolor header
        Array.from(headerRow.querySelectorAll('th')).forEach(th => {
            th.style.background = 'linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%)';
            th.style.color = 'white';
            th.style.fontWeight = '600';
        });
        
        console.log('Header rebuilt successfully!');
    }

    // Load master details on page load if value exists
    window.addEventListener('load', function() {
        loadMasterDetails();
        
        // Template load button for multiple mode
        const loadTemplateBtn2 = document.getElementById('loadTemplateBtn2');
        if (loadTemplateBtn2) {
            loadTemplateBtn2.addEventListener('click', function(e) {
                e.preventDefault();
                loadTemplateForMultipleMode();
            });
        }
        
        // Original template load button for single mode
        document.getElementById('loadTemplateBtn').addEventListener('click', function(e) {
            e.preventDefault();
            loadTemplate();
        });
    });
</script>
@endsection
